import asyncio
import aiohttp
import aiofiles
import os
import json
import urllib.request
import platform
import subprocess
from tqdm import tqdm
import argparse
import xml.etree.ElementTree as ET
import shlex

# === Settings ===
version = "1.21.5"
loader = "vanilla"
base_dir = os.path.abspath("./anymc")
java_path = "java"
username = "Player123"
uuid = "00000000-0000-0000-0000-000000000000"
token = "0"
max_concurrent_downloads = 32
debug = False
offline = False
fabric_url = "https://maven.fabricmc.net/net/fabricmc/fabric-installer/0.11.2/fabric-installer-0.11.2.jar"

# === Variables ===
libraries_dir = os.path.join(base_dir, "libraries")
versions_dir = os.path.join(base_dir, "versions")
game_dir = base_dir
assets_dir = os.path.join(base_dir, "assets")
version_manifest_url = "https://piston-meta.mojang.com/mc/game/version_manifest_v2.json"

# === Fabric Installer ===

fabric_supported_versions = {
    "1.21": "https://maven.fabricmc.net/net/fabricmc/fabric-installer/0.11.2/fabric-installer-0.11.2.jar",
    "1.20.4": "https://maven.fabricmc.net/net/fabricmc/fabric-installer/0.11.2/fabric-installer-0.11.2.jar",
    "1.20.1": "https://maven.fabricmc.net/net/fabricmc/fabric-installer/0.11.1/fabric-installer-0.11.1.jar",
    "1.19.4": "https://maven.fabricmc.net/net/fabricmc/fabric-installer/0.11.0/fabric-installer-0.11.0.jar",
    "1.19.3": "https://maven.fabricmc.net/net/fabricmc/fabric-installer/0.10.2/fabric-installer-0.10.2.jar",
    "1.19.2": "https://maven.fabricmc.net/net/fabricmc/fabric-installer/0.10.2/fabric-installer-0.10.2.jar",
    "1.18.2": "https://maven.fabricmc.net/net/fabricmc/fabric-installer/0.10.2/fabric-installer-0.10.2.jar",
    "1.18.1": "https://maven.fabricmc.net/net/fabricmc/fabric-installer/0.9.1/fabric-installer-0.9.1.jar",
    "1.17.1": "https://maven.fabricmc.net/net/fabricmc/fabric-installer/0.9.2/fabric-installer-0.9.2.jar",
    "1.16.5": "https://maven.fabricmc.net/net/fabricmc/fabric-installer/0.8.2/fabric-installer-0.8.2.jar",
    "1.16.4": "https://maven.fabricmc.net/net/fabricmc/fabric-installer/0.7.4/fabric-installer-0.7.4.jar",
    "1.15.2": "https://maven.fabricmc.net/net/fabricmc/fabric-installer/0.6.1.51/fabric-installer-0.6.1.51.jar",
    "1.14.4": "https://maven.fabricmc.net/net/fabricmc/fabric-installer/0.4.0.32/fabric-installer-0.4.0.32.jar"
}

def install_fabric(mc_version):
    if mc_version not in fabric_supported_versions:
        print("[X] Fabric does not support that version with a hardcoded installer.")
        print("[i] Supported versions:")
        for ver in fabric_supported_versions.keys():
            print(f"  - {ver}")
        mc_version = input("[>] Please pick a supported Minecraft version > ")
        if mc_version not in fabric_supported_versions:
            raise ValueError(f"Unsupported version: {mc_version}")

    installer_url = fabric_supported_versions[mc_version]
    installer_path = os.path.join(base_dir, "fabric-installer.jar")

    print(f"[+] Downloading Fabric installer for {mc_version}...")
    urllib.request.urlretrieve(installer_url, installer_path)

    print("[+] Running Fabric installer...")
    subprocess.run([
        java_path, "-jar", installer_path,
        "client", "-dir", base_dir,
        "-mcversion", mc_version,
        "-noprofile"
    ])

    return f"fabric-loader-{mc_version}"

# === Async Asset Download ===
async def download_asset(session, sem, name, hash, objects_dir, progress_bar):
    subdir = hash[:2]
    url = f"https://resources.download.minecraft.net/{subdir}/{hash}"
    dest = os.path.join(objects_dir, subdir, hash)

    if os.path.isfile(dest):
        progress_bar.update(1)
        return

    async with sem:
        try:
            async with session.get(url) as resp:
                if resp.status == 200:
                    os.makedirs(os.path.dirname(dest), exist_ok=True)
                    f = await aiofiles.open(dest, mode='wb')
                    await f.write(await resp.read())
                    await f.close()
        except Exception as e:
            print(f"[!] Failed to download {name}: {e}")
    progress_bar.update(1)

async def download_all_assets(asset_index_path, objects_dir):
    with open(asset_index_path, "r") as f:
        asset_data = json.load(f)
    objects = asset_data["objects"]

    sem = asyncio.Semaphore(max_concurrent_downloads)
    progress_bar = tqdm(total=len(objects), desc="Downloading assets", unit="file")

    async with aiohttp.ClientSession() as session:
        tasks = [
            download_asset(session, sem, name, info["hash"], objects_dir, progress_bar)
            for name, info in objects.items()
        ]
        await asyncio.gather(*tasks)

    progress_bar.close()

async def download_library(session, sem, artifact, lib_path, progress_bar):
    url = artifact["url"]
    if os.path.isfile(lib_path):
        progress_bar.update(1)
        return

    async with sem:
        try:
            async with session.get(url) as resp:
                if resp.status == 200:
                    os.makedirs(os.path.dirname(lib_path), exist_ok=True)
                    async with aiofiles.open(lib_path, 'wb') as f:
                        await f.write(await resp.read())
        except Exception as e:
            print(f"[!] Failed to download library: {url}\n    {e}")
    progress_bar.update(1)

async def download_all_libraries(version_data, libraries_dir):
    libs = []

    for lib in version_data["libraries"]:
        if "downloads" not in lib:
            continue

        downloads = lib["downloads"]

        # Add main artifact
        if "artifact" in downloads:
            artifact = downloads["artifact"]
            lib_path = os.path.join(libraries_dir, artifact["path"])
            libs.append((artifact, lib_path))

        # Add optional classifiers (e.g. log4j-slf4j2-impl)
        if "classifiers" in downloads:
            for classifier in downloads["classifiers"].values():
                lib_path = os.path.join(libraries_dir, classifier["path"])
                libs.append((classifier, lib_path))

    # Deduplicate by path to avoid downloading the same file twice
    unique_libs = list({lib_path: (artifact, lib_path) for artifact, lib_path in libs}.values())

    sem = asyncio.Semaphore(max_concurrent_downloads)
    progress_bar = tqdm(total=len(unique_libs), desc="Downloading libraries", unit="file")

    async with aiohttp.ClientSession() as session:
        tasks = [
            download_library(session, sem, artifact, lib_path, progress_bar)
            for artifact, lib_path in unique_libs
        ]
        await asyncio.gather(*tasks)

    progress_bar.close()

    # Return full classpath including all jars
    return [lib_path for _, lib_path in unique_libs]

def get_values():
    version = input("[>] Minecraft version: > ")
    pre_loader = input("[>] Use Fabric? [Y/n]> ")
    if pre_loader.lower() == "n":
        loader = "vanilla"
    else:
        loader = "fabric"
    username = input("[>] Username: > ")
    base_dir = os.path.abspath(f"./anymc/{loader}-{version}")
    return version, loader, username, base_dir

# === Main ===
async def main():
    global version
    if offline:
        xml_path = os.path.abspath(f"./anymc/{loader}-{version}/anymc.xml")
        if os.path.exists(xml_path):
            tree = ET.parse(xml_path)
            root = tree.getroot()
            if debug:
                print("[-] XML Root Tag:", root.tag)  # should be 'config'
                print("[-] Children:", [child.tag for child in root])
            cmd_element = root.find("./cmd")
            if cmd_element is not None:
                # Safely split the command string back into list format
                cmd = [arg.text for arg in cmd_element.findall("arg") if arg.text]

                print(f"[+] Launching Minecraft {loader}-{version} using cached command...")
                if debug:
                    print("[-] Finished Command:")
                    print(cmd)

                subprocess.run(cmd, cwd=game_dir)
                return
            else:
                print("[X] Couldn't find launch command in config. Regenerate using online mode.")
                return
        else:
            print("[X] Couldn't find config file. Is this version installed?")
            return

    print(f"[+] Fetching version manifest... (this won't take long)")
    with urllib.request.urlopen(version_manifest_url) as f:
        manifest = json.load(f)

    version_info = next((v for v in manifest["versions"] if v["id"] == version), None)
    if not version_info:
        raise Exception(f"Version {version} not found in manifest.")

    with urllib.request.urlopen(version_info["url"]) as f:
        version_data = json.load(f)

    client_jar_url = version_data["downloads"]["client"]["url"]
    client_jar_path = os.path.join(versions_dir, version, f"{version}.jar")
    os.makedirs(os.path.dirname(client_jar_path), exist_ok=True)
    if not os.path.isfile(client_jar_path):
        print(f"[+] Downloading client.jar...")
        urllib.request.urlretrieve(client_jar_url, client_jar_path)

    print(f"[+] Downloading libraries...")
    """
    classpath = []
    for lib in version_data["libraries"]:
        if "downloads" not in lib or "artifact" not in lib["downloads"]:
            continue
        artifact = lib["downloads"]["artifact"]
        lib_path = os.path.join(libraries_dir, artifact["path"])
        if not os.path.isfile(lib_path):
            os.makedirs(os.path.dirname(lib_path), exist_ok=True)
            urllib.request.urlretrieve(artifact["url"], lib_path)
        classpath.append(lib_path)
    classpath.append(client_jar_path)
    """
    classpath = await download_all_libraries(version_data, libraries_dir)
    classpath.append(client_jar_path)

    asset_index_info = version_data["assetIndex"]
    asset_index_path = os.path.join(assets_dir, "indexes", f"{asset_index_info['id']}.json")
    os.makedirs(os.path.dirname(asset_index_path), exist_ok=True)
    if not os.path.isfile(asset_index_path):
        print(f"[+] Downloading asset index...")
        urllib.request.urlretrieve(asset_index_info["url"], asset_index_path)

    print("[+] Downloading assets...")
    objects_dir = os.path.join(assets_dir, "objects")
    await download_all_assets(asset_index_path, objects_dir)

    natives_dir = os.path.join(base_dir, "natives")
    main_class = version_data["mainClass"]
    log_dir = os.path.join(base_dir, "logs")
    os.makedirs(log_dir, exist_ok=True)

    game_args = [
        "--username", username,
        "--version", version,
        "--gameDir", game_dir,
        "--assetsDir", assets_dir,
        "--assetIndex", version_data["assetIndex"]["id"],
        "--uuid", uuid,
        "--accessToken", token,
        "--userType", "legacy",
        "--versionType", "release"
    ]

    jvm_args = []
    for entry in version_data.get("arguments", {}).get("jvm", []):
        if isinstance(entry, str):
            jvm_args.append(entry.replace("${natives_directory}", natives_dir))
        elif isinstance(entry, dict):
            rules = entry.get("rules", [])
            value = entry.get("value")
            allowed = True
            for rule in rules:
                os_rule = rule.get("os", {}).get("name")
                action = rule.get("action")
                system = platform.system().lower()
                if os_rule and os_rule != system:
                    allowed = (action != "allow")
                elif action == "disallow" and os_rule == system:
                    allowed = False
            if allowed:
                if isinstance(value, str):
                    jvm_args.append(value.replace("${natives_directory}", natives_dir))
                elif isinstance(value, list):
                    jvm_args.extend(arg.replace("${natives_directory}", natives_dir) if isinstance(arg, str) else arg for arg in value)

    jvm_args.append(f"-DlogDir={log_dir}")
    jvm_args += ["-cp", os.pathsep.join(classpath)]
    cmd = [java_path] + jvm_args + [main_class] + game_args

    if loader == "fabric":
        print("[+] Installing Fabric...")
        install_fabric(version)

    print(f"[+] Creating/updating config...")
    config_root = ET.Element("config")
    config_cmd = ET.SubElement(config_root, "cmd")
    for arg in cmd:
        arg_element = ET.SubElement(config_cmd, "arg")
        arg_element.text = arg
    config_ver = ET.SubElement(config_root, "version")
    config_ver.text = version
    config_loader = ET.SubElement(config_root, "loader")
    config_loader.text = loader
    config_tree = ET.ElementTree(config_root)
    config_tree.write(f"{base_dir}\\anymc.xml")

    print(f"[+] Launching Minecraft {loader}-{version}...")
    if debug:
        print("[-] Finished Command:")
        print(cmd)
    subprocess.run(cmd, cwd=game_dir)

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="GUI launcher for AnyMC")
    parser.add_argument("-d", "--debug", action="store_true", help="Enable debug mode")
    parser.add_argument("-o", "--offline", action="store_true", help="Do not try to download Minecraft")
    args = parser.parse_args()
    debug = args.debug
    offline = args.offline
    version, loader, username, base_dir = get_values()
    # Redefine variables after modifying
    libraries_dir = os.path.join(base_dir, "libraries")
    versions_dir = os.path.join(base_dir, "versions")
    game_dir = base_dir
    assets_dir = os.path.join(base_dir, "assets")
    asyncio.run(main())


"""
~~~~~~~~~~~~~~~
AnyMC Mini
~~~~~~~~~~~~~~~
"""

class Settings():
    def __init__(self):
        self.version = "1.21.5"
        self.loader = "vanilla"
        self.username = "Player123"
        self.base_dir = os.path.abspath("./anymc")
        self.java_path = "java"
        self.username = "Player123"
        self.uuid = "00000000-0000-0000-0000-000000000000"
        self.token = "0"
        self.libraries_dir = os.path.join(self.base_dir, "libraries")
        self.versions_dir = os.path.join(self.base_dir, "versions")
        self.game_dir = base_dir
        self.assets_dir = os.path.join(self.base_dir, "assets")
    def export(self):
        default_path = os.path.abspath("./anymc")
        if self.base_dir == default_path:
            self.base_dir = os.path.abspath(f"./anymc/{self.loader}-{self.version}")
        self.libraries_dir = os.path.join(self.base_dir, "libraries")
        self.versions_dir = os.path.join(self.base_dir, "versions")
        self.game_dir = base_dir
        self.assets_dir = os.path.join(self.base_dir, "assets")
        return [self.version, self.loader, self.username, self.base_dir, self.java_path, self.username, self.uuid, self.token, self.libraries_dir, self.versions_dir, self.assets_dir]